<?php

namespace Centraltest\LaravelSeeder\Console;

use Illuminate\Console\Command;
use Centraltest\LaravelSeeder\Scenario\Scenario;

/**
 * Artisan command to execute a seeder scenario.
 */
abstract class SeedScenarioCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'e2e:seed {scenario : Fully qualified scenario class name} {--fresh : Run fresh steps before seeding}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Execute a Laravel Seeder scenario';

    /**
     * @return @return array<string, class-string<\Centraltest\LaravelSeeder\Scenario\Scenario>>
     */
    abstract protected function scenarios(): array;

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        //Validate command
        $scenario = $this->validate();
        if(empty($scenario)) {
            return Command::FAILURE;
        }

        //Run scenario
        $this->runScenario($scenario);

        //Return
        return self::SUCCESS;
    }

    /**
     * Get scenario instance
     *
     * @param string $name
     * @param bool $isFresh
     * @return \Centraltest\LaravelSeeder\Scenario\Scenario|null
     */
    protected function getScenario(string $name, bool $isFresh): ?Scenario
    {
        //Get scenario
        $scenarios = $this->scenarios();
        $scenarioClass = $scenarios[$name] ?? null;

        //Return scenario instance
        if(empty($scenarioClass)) {
            return null;
        }
        return new $scenarioClass($isFresh);
    }

    /**
     * Validate the call
     * 
     * @return void
     */
    protected function validate(): ?Scenario
    {
        //Can be run only in local mode
        if(app()->environment() !== 'local') {
            $this->error('Not local environnement');
            return null;
        }

        //Get scenario
        $scenario = $this->getScenario($this->argument('scenario'), $this->option('fresh'));
        if(empty($scenario)) {
            $this->error("Invalid scenario: {$scenario}. Allowed: ".implode(', ', $this->scenarios()));
            return null;
        }

        //Return
        return $scenario;
    }

    /**
     * Run the scenario
     *
     * @param Scenario $scenario
     * 
     * @return void
     */
    protected function runScenario(Scenario $scenario): void
    {        
        try {
            //Run scenario
            $scenario->run();
        } catch (\Exception $ex) {
            $this->error($ex->getMessage());
        }
    }
}
