<?php

namespace Centraltest\LaravelSeeder\Step;

/**
 * Base class for scenario steps
 */
abstract class ScenarioStep
{
    protected array $data = [];
    protected ?string $identifier = null;
    protected ?array $dataRefCache = [];
    protected ?array $result = null;

    /**
     * Execute the scenario step
     *
     * @return void
     */
    abstract public function run(): void;

    /**
     * Constructor
     *
     * @param array $data
     * @param string|null $identifier
     */
    public function __construct(array $data, ?string $identifier = null)
    {
        $this->data = $data;
        $this->identifier = $identifier;
    }

    /**
     * Set reference cache config
     *
     * @param array $dataRefCache
     *
     * @return self
     */
    public function refCache(array $dataRefCache): self
    {
        $this->dataRefCache = $dataRefCache;
        return $this;
    }

    /**
     * Get identifier
     *
     * @return string|null
     */
    public function identifier(): ?string
    {
        return $this->identifier;
    }

    /**
     * Get result
     *
     * @return array|null
     */
    public function result(): ?array
    {
        return $this->result;
    }
    
    /**
     * Update data from scenario cache
     *
     * @param array $scenarioCache
     *
     * @return self
     */
    public function updateDataFromCache(array $scenarioCache): self
    {
        if(empty($this->dataRefCache)) {
            //Nothing to do
            return $this;
        }

        // Add data from cache to data
        foreach($this->dataRefCache as $field => $cacheConfig) {
            [$cacheRefName, $cacheRefField] = $cacheConfig;

            if($this->isMass()) {
                //Mass insert
                foreach($this->data as &$dataItem) {
                    $dataItem[$field] = $scenarioCache[$cacheRefName][$cacheRefField];
                }
            } else {
                //Single insert
                $this->data[$field] = $scenarioCache[$cacheRefName][$cacheRefField];
            }
        }

        // Return
        return $this;
    }

    /**
     * Create step from array config
     *
     * @param array $config
     *
     * @return self
     */
    public static function fromArray(array $config): self
    {
        $data = $config['data'] ?? [];
        $identifier = $config['identifier'] ?? null;

        if(isset($config['model'])) {
            $step = new ScenarioModelStep($config['model'], $data, $identifier);
        } elseif (isset($config['table'])) {
            $step = new ScenarioTableStep($config['table'], $data, $identifier);
        } elseif(isset($config['factory'])) {
            $step = new FactoryStep($config['factory'], $data, $identifier);
        } else {
            throw new \InvalidArgumentException('Invalid step configuration: missing model or table');
        }

        if (isset($config['data_cache'])) {
            $step->refCache($config['data_cache']);
        }

        return $step;
    }

    /**
     * Check if data is mass insert
     *
     * @return bool
     */
    private function isMass(): bool
    {
        return array_keys($this->data) === range(0, count($this->data) - 1);
    }
}
